<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

if(!isActiveUser()) {
    header("Location: activate.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Get filter parameters
$status_filter = isset($_GET['status']) ? sanitize($_GET['status']) : '';
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Build query
$where_conditions = ["o.user_id = $user_id"];
$params = [];
$types = "";

if($status_filter && $status_filter != 'all') {
    $where_conditions[] = "o.status = ?";
    $params[] = $status_filter;
    $types .= "s";
}

if($search) {
    $where_conditions[] = "(p.name LIKE ? OR o.customer_name LIKE ? OR o.telegram_username LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $types .= "sss";
}

$where_sql = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total orders
$count_sql = "SELECT COUNT(*) as total FROM orders o 
              JOIN products p ON o.product_id = p.id 
              $where_sql";
$count_stmt = $conn->prepare($count_sql);
if($types) $count_stmt->bind_param($types, ...$params);
$count_stmt->execute();
$total_result = $count_stmt->get_result();
$total_orders = $total_result->fetch_assoc()['total'];
$total_pages = ceil($total_orders / $per_page);

// Get orders
$orders_sql = "SELECT o.*, p.name as product_name, p.image_url, p.sale_price 
               FROM orders o 
               JOIN products p ON o.product_id = p.id 
               $where_sql 
               ORDER BY o.created_at DESC 
               LIMIT ? OFFSET ?";
$params[] = $per_page;
$params[] = $offset;
$types .= "ii";

$orders_stmt = $conn->prepare($orders_sql);
$orders_stmt->bind_param($types, ...$params);
$orders_stmt->execute();
$orders_result = $orders_stmt->get_result();

// Get status counts
$status_counts = $conn->query("
    SELECT status, COUNT(*) as count 
    FROM orders 
    WHERE user_id = $user_id 
    GROUP BY status
")->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Orders - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        .order-card {
            transition: all 0.3s ease;
        }
        .order-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
        }
        .status-badge {
            transition: all 0.2s ease;
        }
        .filter-tab {
            transition: all 0.3s ease;
        }
        .filter-tab:hover {
            transform: scale(1.05);
        }
        .order-details {
            max-height: 0;
            overflow: hidden;
            transition: max-height 0.5s ease;
        }
        .order-details.active {
            max-height: 500px;
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Include Header -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Breadcrumb -->
            <div class="mb-6">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="index.php" class="inline-flex items-center text-sm text-gray-700 hover:text-blue-600">
                                <i class="fas fa-home mr-2"></i> Home
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <a href="dashboard.php" class="ml-1 text-sm text-gray-700 hover:text-blue-600 md:ml-2">Dashboard</a>
                            </div>
                        </li>
                        <li aria-current="page">
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <span class="ml-1 text-sm text-gray-500 md:ml-2 font-medium">My Orders</span>
                            </div>
                        </li>
                    </ol>
                </nav>
            </div>
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-shopping-bag mr-2"></i> My Orders
                </h1>
                <p class="text-gray-600">Track and manage your purchases</p>
            </div>
            
            <!-- Stats Cards -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                <?php
                $status_data = [
                    'all' => ['color' => 'blue', 'icon' => 'fa-boxes', 'count' => $total_orders],
                    'pending' => ['color' => 'yellow', 'icon' => 'fa-clock', 'count' => 0],
                    'processing' => ['color' => 'blue', 'icon' => 'fa-cogs', 'count' => 0],
                    'completed' => ['color' => 'green', 'icon' => 'fa-check-circle', 'count' => 0]
                ];
                
                foreach($status_counts as $status) {
                    if(isset($status_data[$status['status']])) {
                        $status_data[$status['status']]['count'] = $status['count'];
                    }
                }
                
                foreach($status_data as $status => $data):
                ?>
                <a href="my-orders.php?status=<?php echo $status == 'all' ? '' : $status; ?>" 
                   class="filter-tab">
                    <div class="bg-white rounded-xl shadow-lg p-6 text-center hover:shadow-xl transition duration-300 <?php echo $status_filter == $status || ($status == 'all' && !$status_filter) ? 'ring-2 ring-' . $data['color'] . '-500' : ''; ?>">
                        <div class="w-16 h-16 bg-<?php echo $data['color']; ?>-100 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas <?php echo $data['icon']; ?> text-<?php echo $data['color']; ?>-600 text-2xl"></i>
                        </div>
                        <h3 class="text-3xl font-bold text-gray-800 mb-2"><?php echo $data['count']; ?></h3>
                        <p class="text-gray-600 font-medium capitalize">
                            <?php echo str_replace('_', ' ', $status); ?> Orders
                        </p>
                    </div>
                </a>
                <?php endforeach; ?>
            </div>
            
            <!-- Search and Filter -->
            <div class="bg-white rounded-xl shadow-lg p-6 mb-8 animate__animated animate__fadeIn">
                <form method="GET" action="" class="space-y-4">
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <!-- Status Filter -->
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                <i class="fas fa-filter text-blue-500 mr-2"></i> Status
                            </label>
                            <select name="status" 
                                    class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300">
                                <option value="">All Status</option>
                                <option value="pending" <?php echo $status_filter == 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="processing" <?php echo $status_filter == 'processing' ? 'selected' : ''; ?>>Processing</option>
                                <option value="completed" <?php echo $status_filter == 'completed' ? 'selected' : ''; ?>>Completed</option>
                                <option value="cancelled" <?php echo $status_filter == 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                            </select>
                        </div>
                        
                        <!-- Search -->
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                <i class="fas fa-search text-blue-500 mr-2"></i> Search
                            </label>
                            <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                   placeholder="Product name, customer name...">
                        </div>
                        
                        <!-- Date Range -->
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                <i class="fas fa-calendar text-blue-500 mr-2"></i> Sort By
                            </label>
                            <select class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300">
                                <option>Newest First</option>
                                <option>Oldest First</option>
                                <option>Price: High to Low</option>
                                <option>Price: Low to High</option>
                            </select>
                        </div>
                    </div>
                    
                    <!-- Action Buttons -->
                    <div class="flex flex-wrap gap-3">
                        <button type="submit" 
                                class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300 transform hover:scale-105">
                            <i class="fas fa-search mr-2"></i> Apply Filters
                        </button>
                        
                        <?php if($status_filter || $search): ?>
                            <a href="my-orders.php" 
                               class="bg-gray-200 text-gray-700 font-bold py-3 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                                <i class="fas fa-times mr-2"></i> Clear Filters
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
            
            <!-- Orders List -->
            <div class="bg-white rounded-xl shadow-lg overflow-hidden">
                <div class="px-6 py-4 border-b bg-gradient-to-r from-blue-50 to-indigo-50">
                    <div class="flex items-center justify-between">
                        <h3 class="text-xl font-bold text-gray-800">
                            <i class="fas fa-list mr-2"></i> Orders (<?php echo $total_orders; ?>)
                        </h3>
                        <span class="text-gray-600 text-sm">
                            Page <?php echo $page; ?> of <?php echo max(1, $total_pages); ?>
                        </span>
                    </div>
                </div>
                
                <div class="p-6">
                    <?php if($orders_result->num_rows > 0): ?>
                        <div class="space-y-6">
                            <?php while($order = $orders_result->fetch_assoc()): 
                                $status_color = match($order['status']) {
                                    'pending' => 'yellow',
                                    'processing' => 'blue',
                                    'completed' => 'green',
                                    'cancelled' => 'red',
                                    default => 'gray'
                                };
                                
                                $status_icon = match($order['status']) {
                                    'pending' => 'fa-clock',
                                    'processing' => 'fa-cogs',
                                    'completed' => 'fa-check-circle',
                                    'cancelled' => 'fa-times-circle',
                                    default => 'fa-question-circle'
                                };
                            ?>
                            <div class="order-card bg-gray-50 rounded-xl overflow-hidden">
                                <!-- Order Header -->
                                <div class="p-6">
                                    <div class="flex flex-col md:flex-row md:items-center justify-between mb-4">
                                        <div class="mb-4 md:mb-0">
                                            <div class="flex items-center mb-2">
                                                <span class="text-lg font-bold text-gray-800">
                                                    Order #<?php echo str_pad($order['id'], 6, '0', STR_PAD_LEFT); ?>
                                                </span>
                                                <span class="status-badge ml-4 px-3 py-1 rounded-full text-sm font-bold bg-<?php echo $status_color; ?>-100 text-<?php echo $status_color; ?>-800">
                                                    <i class="fas <?php echo $status_icon; ?> mr-1"></i>
                                                    <?php echo ucfirst($order['status']); ?>
                                                </span>
                                            </div>
                                            <p class="text-gray-600 text-sm">
                                                <i class="far fa-calendar mr-1"></i>
                                                <?php echo date('F d, Y h:i A', strtotime($order['created_at'])); ?>
                                            </p>
                                        </div>
                                        
                                        <div class="text-right">
                                            <p class="text-2xl font-bold text-gray-800">
                                                ৳<?php echo number_format($order['amount'], 2); ?>
                                            </p>
                                            <p class="text-gray-600 text-sm">
                                                Paid via Wallet
                                            </p>
                                        </div>
                                    </div>
                                    
                                    <!-- Order Content -->
                                    <div class="flex flex-col md:flex-row items-start">
                                        <!-- Product Info -->
                                        <div class="flex-1">
                                            <div class="flex items-start">
                                                <div class="w-20 h-20 rounded-lg overflow-hidden mr-4">
                                                    <img src="<?php echo $order['image_url'] ?: 'https://via.placeholder.com/100'; ?>" 
                                                         alt="<?php echo htmlspecialchars($order['product_name']); ?>"
                                                         class="w-full h-full object-cover">
                                                </div>
                                                <div>
                                                    <h4 class="font-bold text-gray-800 text-lg mb-1">
                                                        <?php echo htmlspecialchars($order['product_name']); ?>
                                                    </h4>
                                                    <p class="text-gray-600 text-sm mb-2">
                                                        Purchased for ৳<?php echo number_format($order['sale_price'], 2); ?>
                                                    </p>
                                                    <div class="flex flex-wrap gap-2">
                                                        <?php if($order['telegram_username']): ?>
                                                        <span class="inline-flex items-center px-2 py-1 bg-blue-100 text-blue-700 rounded text-xs">
                                                            <i class="fab fa-telegram mr-1"></i> <?php echo $order['telegram_username']; ?>
                                                        </span>
                                                        <?php endif; ?>
                                                        
                                                        <?php if($order['whatsapp_number']): ?>
                                                        <span class="inline-flex items-center px-2 py-1 bg-green-100 text-green-700 rounded text-xs">
                                                            <i class="fab fa-whatsapp mr-1"></i> <?php echo $order['whatsapp_number']; ?>
                                                        </span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Actions -->
                                        <div class="mt-4 md:mt-0 md:ml-6">
                                            <button onclick="toggleOrderDetails(<?php echo $order['id']; ?>)" 
                                                    class="bg-white border border-gray-300 text-gray-700 font-medium py-2 px-4 rounded-lg hover:bg-gray-50 transition duration-300">
                                                <i class="fas fa-eye mr-2"></i> View Details
                                            </button>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Order Details (Hidden by default) -->
                                <div id="details-<?php echo $order['id']; ?>" class="order-details border-t">
                                    <div class="p-6">
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                            <!-- Order Information -->
                                            <div>
                                                <h4 class="font-bold text-gray-800 mb-4">
                                                    <i class="fas fa-info-circle mr-2"></i> Order Information
                                                </h4>
                                                <div class="space-y-3">
                                                    <div class="flex justify-between">
                                                        <span class="text-gray-600">Order ID:</span>
                                                        <span class="font-medium">#<?php echo str_pad($order['id'], 6, '0', STR_PAD_LEFT); ?></span>
                                                    </div>
                                                    <div class="flex justify-between">
                                                        <span class="text-gray-600">Order Date:</span>
                                                        <span class="font-medium"><?php echo date('F d, Y h:i A', strtotime($order['created_at'])); ?></span>
                                                    </div>
                                                    <div class="flex justify-between">
                                                        <span class="text-gray-600">Last Updated:</span>
                                                        <span class="font-medium"><?php echo date('F d, Y h:i A', strtotime($order['updated_at'])); ?></span>
                                                    </div>
                                                    <div class="flex justify-between">
                                                        <span class="text-gray-600">Payment Method:</span>
                                                        <span class="font-medium">Wallet Balance</span>
                                                    </div>
                                                    <div class="flex justify-between">
                                                        <span class="text-gray-600">Payment Status:</span>
                                                        <span class="font-medium text-green-600">Paid</span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <!-- Customer Information -->
                                            <div>
                                                <h4 class="font-bold text-gray-800 mb-4">
                                                    <i class="fas fa-user mr-2"></i> Customer Information
                                                </h4>
                                                <div class="space-y-3">
                                                    <div class="flex justify-between">
                                                        <span class="text-gray-600">Name:</span>
                                                        <span class="font-medium"><?php echo $order['customer_name']; ?></span>
                                                    </div>
                                                    <?php if($order['telegram_username']): ?>
                                                    <div class="flex justify-between">
                                                        <span class="text-gray-600">Telegram:</span>
                                                        <a href="https://t.me/<?php echo ltrim($order['telegram_username'], '@'); ?>" 
                                                           target="_blank" 
                                                           class="font-medium text-blue-600 hover:text-blue-700">
                                                            <?php echo $order['telegram_username']; ?>
                                                        </a>
                                                    </div>
                                                    <?php endif; ?>
                                                    
                                                    <?php if($order['whatsapp_number']): ?>
                                                    <div class="flex justify-between">
                                                        <span class="text-gray-600">WhatsApp:</span>
                                                        <a href="https://wa.me/<?php echo preg_replace('/[^0-9]/', '', $order['whatsapp_number']); ?>" 
                                                           target="_blank" 
                                                           class="font-medium text-green-600 hover:text-green-700">
                                                            <?php echo $order['whatsapp_number']; ?>
                                                        </a>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Admin Message -->
                                        <?php if(!empty($order['admin_message'])): ?>
                                        <div class="mt-6 pt-6 border-t">
                                            <h4 class="font-bold text-gray-800 mb-3">
                                                <i class="fas fa-comment-alt mr-2"></i> Admin Message
                                            </h4>
                                            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                                <p class="text-gray-800"><?php echo nl2br(htmlspecialchars($order['admin_message'])); ?></p>
                                                <p class="text-gray-600 text-sm mt-2">
                                                    <i class="fas fa-clock mr-1"></i> Last updated: <?php echo date('F d, Y', strtotime($order['updated_at'])); ?>
                                                </p>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <!-- Actions -->
                                        <div class="mt-6 pt-6 border-t">
                                            <div class="flex flex-wrap gap-3">
                                                <?php if($order['status'] == 'pending'): ?>
                                                    <button onclick="cancelOrder(<?php echo $order['id']; ?>)" 
                                                            class="bg-red-100 text-red-700 font-bold py-2 px-4 rounded-lg hover:bg-red-200 transition duration-300">
                                                        <i class="fas fa-times mr-2"></i> Cancel Order
                                                    </button>
                                                <?php endif; ?>
                                                
                                                <a href="https://t.me/support" target="_blank" 
                                                   class="bg-blue-100 text-blue-700 font-bold py-2 px-4 rounded-lg hover:bg-blue-200 transition duration-300">
                                                    <i class="fas fa-headset mr-2"></i> Contact Support
                                                </a>
                                                
                                                <a href="product-details.php?id=<?php echo $order['product_id']; ?>" 
                                                   class="bg-gray-100 text-gray-700 font-bold py-2 px-4 rounded-lg hover:bg-gray-200 transition duration-300">
                                                    <i class="fas fa-eye mr-2"></i> View Product
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endwhile; ?>
                        </div>
                        
                        <!-- Pagination -->
                        <?php if($total_pages > 1): ?>
                        <div class="mt-8 flex justify-center">
                            <nav class="flex items-center space-x-2">
                                <?php if($page > 1): ?>
                                    <a href="my-orders.php?page=<?php echo $page-1; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>"
                                       class="w-10 h-10 flex items-center justify-center rounded-lg bg-white border border-gray-300 hover:bg-gray-50 transition duration-300">
                                        <i class="fas fa-chevron-left"></i>
                                    </a>
                                <?php endif; ?>
                                
                                <?php 
                                $start = max(1, $page - 2);
                                $end = min($total_pages, $page + 2);
                                
                                for($i = $start; $i <= $end; $i++): 
                                ?>
                                    <a href="my-orders.php?page=<?php echo $i; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>"
                                       class="w-10 h-10 flex items-center justify-center rounded-lg font-medium transition duration-300
                                              <?php echo $i == $page ? 'bg-blue-600 text-white' : 'bg-white border border-gray-300 hover:bg-gray-50 text-gray-700'; ?>">
                                        <?php echo $i; ?>
                                    </a>
                                <?php endfor; ?>
                                
                                <?php if($page < $total_pages): ?>
                                    <a href="my-orders.php?page=<?php echo $page+1; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>"
                                       class="w-10 h-10 flex items-center justify-center rounded-lg bg-white border border-gray-300 hover:bg-gray-50 transition duration-300">
                                        <i class="fas fa-chevron-right"></i>
                                    </a>
                                <?php endif; ?>
                            </nav>
                        </div>
                        <?php endif; ?>
                        
                    <?php else: ?>
                        <!-- No Orders -->
                        <div class="text-center py-12">
                            <div class="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                                <i class="fas fa-box-open text-gray-400 text-4xl"></i>
                            </div>
                            <h3 class="text-2xl font-bold text-gray-700 mb-3">No Orders Found</h3>
                            <p class="text-gray-600 mb-8 max-w-md mx-auto">
                                <?php echo $search || $status_filter ? 'No orders match your search criteria.' : 'You haven\'t placed any orders yet.'; ?>
                            </p>
                            <div class="flex flex-col sm:flex-row justify-center gap-4">
                                <a href="products.php" 
                                   class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-8 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                                    <i class="fas fa-shopping-cart mr-2"></i> Browse Products
                                </a>
                                <?php if($search || $status_filter): ?>
                                    <a href="my-orders.php" 
                                       class="bg-gray-200 text-gray-700 font-bold py-3 px-8 rounded-lg hover:bg-gray-300 transition duration-300">
                                        <i class="fas fa-redo mr-2"></i> Reset Filters
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Order Status Guide -->
            <div class="mt-8 grid grid-cols-1 md:grid-cols-4 gap-4">
                <div class="bg-white p-4 rounded-lg border-l-4 border-yellow-500">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-yellow-100 rounded-full flex items-center justify-center mr-3">
                            <i class="fas fa-clock text-yellow-600"></i>
                        </div>
                        <div>
                            <h4 class="font-bold text-gray-800">Pending</h4>
                            <p class="text-gray-600 text-sm">Order received, awaiting processing</p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white p-4 rounded-lg border-l-4 border-blue-500">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center mr-3">
                            <i class="fas fa-cogs text-blue-600"></i>
                        </div>
                        <div>
                            <h4 class="font-bold text-gray-800">Processing</h4>
                            <p class="text-gray-600 text-sm">Order is being processed</p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white p-4 rounded-lg border-l-4 border-green-500">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center mr-3">
                            <i class="fas fa-check-circle text-green-600"></i>
                        </div>
                        <div>
                            <h4 class="font-bold text-gray-800">Completed</h4>
                            <p class="text-gray-600 text-sm">Order delivered successfully</p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white p-4 rounded-lg border-l-4 border-red-500">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-red-100 rounded-full flex items-center justify-center mr-3">
                            <i class="fas fa-times-circle text-red-600"></i>
                        </div>
                        <div>
                            <h4 class="font-bold text-gray-800">Cancelled</h4>
                            <p class="text-gray-600 text-sm">Order has been cancelled</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Cancel Order Modal -->
    <div id="cancelModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center p-4">
        <div class="bg-white rounded-2xl max-w-md w-full">
            <div class="p-6">
                <h3 class="text-xl font-bold text-gray-800 mb-4">
                    <i class="fas fa-exclamation-triangle text-yellow-500 mr-2"></i> Cancel Order
                </h3>
                <p class="text-gray-600 mb-6">
                    Are you sure you want to cancel this order? This action cannot be undone.
                </p>
                <form id="cancelForm" method="POST" action="ajax/cancel-order.php" class="space-y-4">
                    <input type="hidden" name="order_id" id="cancelOrderId">
                    
                    <div>
                        <label class="block text-gray-700 text-sm font-bold mb-2">
                            Reason for cancellation
                        </label>
                        <select name="reason" class="w-full px-4 py-3 border border-gray-300 rounded-lg">
                            <option value="">Select a reason</option>
                            <option value="changed_mind">Changed my mind</option>
                            <option value="wrong_product">Ordered wrong product</option>
                            <option value="found_cheaper">Found cheaper elsewhere</option>
                            <option value="technical_issue">Technical issue</option>
                            <option value="other">Other</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-gray-700 text-sm font-bold mb-2">
                            Additional notes (optional)
                        </label>
                        <textarea name="notes" rows="2" 
                                  class="w-full px-4 py-3 border border-gray-300 rounded-lg"
                                  placeholder="Any additional information..."></textarea>
                    </div>
                    
                    <div class="flex justify-end space-x-3 pt-4">
                        <button type="button" onclick="closeCancelModal()" 
                                class="bg-gray-200 text-gray-700 font-bold py-2 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                            Cancel
                        </button>
                        <button type="submit" 
                                class="bg-red-600 text-white font-bold py-2 px-6 rounded-lg hover:bg-red-700 transition duration-300">
                            <i class="fas fa-times mr-2"></i> Confirm Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script>
    // Toggle order details
    function toggleOrderDetails(orderId) {
        const details = document.getElementById('details-' + orderId);
        const button = event.target.closest('button');
        
        if(details.classList.contains('active')) {
            details.classList.remove('active');
            button.innerHTML = '<i class="fas fa-eye mr-2"></i> View Details';
        } else {
            // Close all other open details
            document.querySelectorAll('.order-details.active').forEach(el => {
                if(el.id !== 'details-' + orderId) {
                    el.classList.remove('active');
                    el.previousElementSibling.querySelector('button').innerHTML = '<i class="fas fa-eye mr-2"></i> View Details';
                }
            });
            
            details.classList.add('active');
            button.innerHTML = '<i class="fas fa-eye-slash mr-2"></i> Hide Details';
        }
    }
    
    // Cancel order modal
    function cancelOrder(orderId) {
        document.getElementById('cancelOrderId').value = orderId;
        document.getElementById('cancelModal').classList.remove('hidden');
        document.getElementById('cancelModal').classList.add('flex');
    }
    
    function closeCancelModal() {
        document.getElementById('cancelModal').classList.add('hidden');
        document.getElementById('cancelModal').classList.remove('flex');
    }
    
    // Handle cancel form submission
    document.getElementById('cancelForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        
        fetch(this.action, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if(data.success) {
                showNotification('Order cancelled successfully!', 'success');
                closeCancelModal();
                setTimeout(() => {
                    location.reload();
                }, 1500);
            } else {
                showNotification(data.message || 'Error cancelling order', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Network error', 'error');
        });
    });
    
    function showNotification(message, type) {
        const notification = document.createElement('div');
        notification.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
                <span>${message}</span>
            </div>
        `;
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.classList.remove('animate__fadeInRight');
            notification.classList.add('animate__fadeOutRight');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
    
    // Add animations
    document.addEventListener('DOMContentLoaded', function() {
        const cards = document.querySelectorAll('.order-card');
        cards.forEach((card, index) => {
            setTimeout(() => {
                card.classList.add('animate__animated', 'animate__fadeInUp');
            }, index * 100);
        });
        
        // Close modals when clicking outside
        document.addEventListener('click', function(event) {
            const cancelModal = document.getElementById('cancelModal');
            const cancelContent = cancelModal.querySelector('.bg-white');
            
            if(cancelModal.classList.contains('flex') && !cancelContent.contains(event.target) && !event.target.closest('[onclick*="cancelOrder"]')) {
                closeCancelModal();
            }
        });
    });
    </script>
</body>
</html>